<?php
// interface.php
session_start();
require 'config.php';

// Redirect to login if not authenticated
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// Fetch filter parameters for Transactions
$search = $_GET['search'] ?? '';
$rubbish_type_id = $_GET['rubbish_type_id'] ?? '';
$start_date = $_GET['start_date'] ?? '';
$end_date = $_GET['end_date'] ?? '';

// Fetch search parameter for Pending Payments
$search_pending = $_GET['search_pending'] ?? '';

// Pagination settings for Pending Payments
$pending_limit = 50;
$pending_page = isset($_GET['pending_page']) && is_numeric($_GET['pending_page']) ? (int)$_GET['pending_page'] : 1;
$pending_offset = ($pending_page - 1) * $pending_limit;

// Parameters for Statistics tab
$stats_plate = $_GET['stats_plate'] ?? '';
$stats_start_date = $_GET['stats_start_date'] ?? '';
$stats_end_date = $_GET['stats_end_date'] ?? '';
$stats_start_time = $_GET['stats_start_time'] ?? '';
$stats_end_time = $_GET['stats_end_time'] ?? '';

// Pagination settings for Transactions
$limit = 20; // Records per page
$page = isset($_GET['page']) && is_numeric($_GET['page']) ? (int)$_GET['page'] : 1;
$offset = ($page - 1) * $limit;

// Prepare Transactions Query with Filters and Pagination
$transactionsSql = "SELECT 
                        p.id as payment_id,
                        p.license_plate, 
                        p.weight_diff, 
                        p.total_amount, 
                        p.timestamp as payment_time,
                        r.name as rubbish_type,
                        r.rate_per_kg
                    FROM payments p
                    JOIN rubbish_types r ON p.rubbish_type_id = r.id
                    WHERE p.license_plate LIKE :search";

$bindParams = [
    ':search' => '%' . $search . '%'
];

if (!empty($rubbish_type_id)) {
    $transactionsSql .= " AND p.rubbish_type_id = :rubbish_type_id";
    $bindParams[':rubbish_type_id'] = $rubbish_type_id;
}

if (!empty($start_date)) {
    $transactionsSql .= " AND p.timestamp >= :start_date";
    $bindParams[':start_date'] = $start_date . ' 00:00:00';
}

if (!empty($end_date)) {
    $transactionsSql .= " AND p.timestamp <= :end_date";
    $bindParams[':end_date'] = $end_date . ' 23:59:59';
}

$transactionsSql .= " ORDER BY p.timestamp DESC LIMIT :limit OFFSET :offset";
$transactionsStmt = $pdo->prepare($transactionsSql);
$transactionsStmt->bindValue(':search', '%' . $search . '%', PDO::PARAM_STR);
if (!empty($rubbish_type_id)) {
    $transactionsStmt->bindValue(':rubbish_type_id', $rubbish_type_id, PDO::PARAM_INT);
}
if (!empty($start_date)) {
    $transactionsStmt->bindValue(':start_date', $start_date . ' 00:00:00');
}
if (!empty($end_date)) {
    $transactionsStmt->bindValue(':end_date', $end_date . ' 23:59:59');
}
$transactionsStmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
$transactionsStmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
$transactionsStmt->execute();
$transactionRows = $transactionsStmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch total number of transactions for pagination
$countSql = "SELECT COUNT(*) FROM payments p JOIN rubbish_types r ON p.rubbish_type_id = r.id WHERE p.license_plate LIKE :search";

$countParams = [
    ':search' => '%' . $search . '%'
];

if (!empty($rubbish_type_id)) {
    $countSql .= " AND p.rubbish_type_id = :rubbish_type_id";
    $countParams[':rubbish_type_id'] = $rubbish_type_id;
}

if (!empty($start_date)) {
    $countSql .= " AND p.timestamp >= :start_date";
    $countParams[':start_date'] = $start_date . ' 00:00:00';
}

if (!empty($end_date)) {
    $countSql .= " AND p.timestamp <= :end_date";
    $countParams[':end_date'] = $end_date . ' 23:59:59';
}

$countStmt = $pdo->prepare($countSql);
foreach ($countParams as $key => $value) {
    if ($key === ':rubbish_type_id') {
        $countStmt->bindValue($key, $value, PDO::PARAM_INT);
    } else {
        $countStmt->bindValue($key, $value);
    }
}
$countStmt->execute();
$total = $countStmt->fetchColumn();
$totalPages = ceil($total / $limit);

// Fetch rubbish types for dropdowns
$types = $pdo->query("SELECT * FROM rubbish_types")->fetchAll(PDO::FETCH_ASSOC);
$typeRates = [];
foreach ($types as $t) {
    $typeRates[$t['id']] = (float)$t['rate_per_kg'];
}

// Build Statistics query
$statsSql = "SELECT DATE(timestamp) AS day, COUNT(*) AS count FROM entries WHERE direction='in'";
$statsParams = [];
if (!empty($stats_plate)) {
    $statsSql .= " AND license_plate LIKE :stats_plate";
    $statsParams[':stats_plate'] = '%' . $stats_plate . '%';
}
if (!empty($stats_start_date)) {
    $statsSql .= " AND DATE(timestamp) >= :stats_start_date";
    $statsParams[':stats_start_date'] = $stats_start_date;
}
if (!empty($stats_end_date)) {
    $statsSql .= " AND DATE(timestamp) <= :stats_end_date";
    $statsParams[':stats_end_date'] = $stats_end_date;
}
if (!empty($stats_start_time)) {
    $statsSql .= " AND TIME(timestamp) >= :stats_start_time";
    $statsParams[':stats_start_time'] = $stats_start_time;
}
if (!empty($stats_end_time)) {
    $statsSql .= " AND TIME(timestamp) <= :stats_end_time";
    $statsParams[':stats_end_time'] = $stats_end_time;
}
$statsSql .= " GROUP BY DATE(timestamp) ORDER BY day DESC";
$statsStmt = $pdo->prepare($statsSql);
foreach ($statsParams as $k => $v) {
    $statsStmt->bindValue($k, $v);
}
$statsStmt->execute();
$statsRows = $statsStmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch recent POS transactions
$posSql = "SELECT pt.id, pt.payment_id, pt.http_status, pt.request_json, pt.response_json, pt.created_at, p.license_plate
            FROM pos_transactions pt
            LEFT JOIN payments p ON pt.payment_id = p.id
            ORDER BY pt.created_at DESC LIMIT 50";
$posStmt = $pdo->prepare($posSql);
$posStmt->execute();
$posRows = $posStmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch Pending Payments with pagination and search
$pendingSql = "SELECT
                inE.id as in_id,
                inE.license_plate,
                inE.weight_kg as in_weight,
                inE.timestamp as in_time,
                inE.image1, inE.image2, inE.image3,
                outE.id as out_id,
                outE.weight_kg as out_weight,
                outE.timestamp as out_time,
                outE.out_image,
                outE.out_image2,
                outE.out_image3
            FROM entries inE
            JOIN entries outE
                ON (
                   (
                       inE.license_plate = outE.license_plate
                       AND inE.license_plate <> 'unknown'
                       AND outE.timestamp >= inE.timestamp - INTERVAL 10 SECOND
                   )
                   OR
                   (
                       inE.license_plate = 'unknown'
                       AND outE.license_plate = 'unknown'
                       AND inE.timestamp = outE.timestamp
                   )
                )
                AND inE.direction = 'in'
                AND outE.direction = 'out'
            WHERE inE.processed = 0 AND outE.processed = 0";

$pendingParams = [];
if (!empty($search_pending)) {
    $pendingSql .= " AND inE.license_plate LIKE :search_pending";
    $pendingParams[':search_pending'] = '%' . $search_pending . '%';
}

$pendingSql .= " ORDER BY (inE.license_plate='unknown') DESC, inE.timestamp DESC LIMIT :limit OFFSET :offset";

try {
    $pendingStmt = $pdo->prepare($pendingSql);
    if (!empty($search_pending)) {
        $pendingStmt->bindValue(':search_pending', '%' . $search_pending . '%');
    }
    $pendingStmt->bindValue(':limit', (int)$pending_limit, PDO::PARAM_INT);
    $pendingStmt->bindValue(':offset', (int)$pending_offset, PDO::PARAM_INT);
    $pendingStmt->execute();
    $pendingRows = $pendingStmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    // Log the error and set $pendingRows to an empty array
    error_log("Error fetching pending payments: " . $e->getMessage());
    $pendingRows = [];
}

// Fetch total count for pagination of pending payments
$pendingCountSql = "SELECT COUNT(*) FROM entries inE JOIN entries outE ON (
                       (
                           inE.license_plate = outE.license_plate
                           AND inE.license_plate <> 'unknown'
                           AND outE.timestamp >= inE.timestamp - INTERVAL 10 SECOND
                       )
                       OR
                       (inE.license_plate = 'unknown' AND outE.license_plate = 'unknown' AND inE.timestamp = outE.timestamp)
                   )
                   AND inE.direction = 'in' AND outE.direction = 'out'
                   WHERE inE.processed = 0 AND outE.processed = 0";
if (!empty($search_pending)) {
    $pendingCountSql .= " AND inE.license_plate LIKE :search_pending";
}
$pendingCountStmt = $pdo->prepare($pendingCountSql);
if (!empty($search_pending)) {
    $pendingCountStmt->bindValue(':search_pending', '%' . $search_pending . '%');
}
$pendingCountStmt->execute();
$pending_total = $pendingCountStmt->fetchColumn();
$pending_total_pages = ceil($pending_total / $pending_limit);

$msg = $_GET['msg'] ?? '';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>Management Interface</title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Bootstrap Icons (optional) -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        body {
            padding: 20px;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        .image-thumbnail {
            width: 90px;
            height: 90px;
            object-fit: contain;
            cursor: pointer;
            border: 1px solid #ccc;
            border-radius: 4px;
        }
        /* Modal styles */
        .modal-img {
            width: 100%;
        }
        /* Remove internal table scrollbars */
        .table-responsive {
            overflow: visible;
        }
        .unknown-row {
            background-color: #ffecec;
        }
        .json-cell {
            max-width: 300px;
            white-space: pre-wrap;
            word-break: break-all;
        }
        /* Responsive adjustments */
        @media (max-width: 767.98px) {
            .image-thumbnail {
                width: 60px;
                height: 60px;
            }
        }
    </style>
</head>
<body>
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1>Management Interface</h1>
        <div>
            <span class="me-3">Logged in as: <strong><?php echo htmlspecialchars($_SESSION['username']); ?></strong></span>
            <a href="logout.php" class="btn btn-outline-danger btn-sm">Logout</a>
        </div>
    </div>

    <?php if($msg): ?>
    <div class="alert alert-info alert-dismissible fade show" role="alert">
        <?php echo htmlspecialchars($msg); ?>
        <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
    </div>
    <?php endif; ?>

    <!-- Tabs Navigation -->
    <ul class="nav nav-tabs" id="managementTab" role="tablist">
        <li class="nav-item" role="presentation">
            <button class="nav-link active" id="pending-tab" data-bs-toggle="tab" data-bs-target="#pending" type="button" role="tab" aria-controls="pending" aria-selected="true">Pending Payments</button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="transactions-tab" data-bs-toggle="tab" data-bs-target="#transactions" type="button" role="tab" aria-controls="transactions" aria-selected="false">Transactions</button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="pos-trans-tab" data-bs-toggle="tab" data-bs-target="#pos-trans" type="button" role="tab" aria-controls="pos-trans" aria-selected="false">POS Transactions</button>
        </li>
        <li class="nav-item" role="presentation">
            <button class="nav-link" id="stats-tab" data-bs-toggle="tab" data-bs-target="#stats" type="button" role="tab" aria-controls="stats" aria-selected="false">Statistics</button>
        </li>
    </ul>
    <div class="tab-content" id="managementTabContent">
        <!-- Pending Payments Tab -->
        <div class="tab-pane fade show active" id="pending" role="tabpanel" aria-labelledby="pending-tab">
            <div class="mt-3 mb-3">
                <form method="get" action="interface.php" class="d-flex">
                    <input type="text" name="search_pending" class="form-control me-2" placeholder="Search by License Plate" value="<?php echo htmlspecialchars($search_pending); ?>">
                    <button type="submit" class="btn btn-outline-success">Search</button>
                    <a href="interface.php" class="btn btn-secondary ms-2">Reset</a>
                </form>
            </div>
            <div class="table-responsive">
                <table class="table table-bordered table-hover align-middle" id="pendingPaymentsTable">
                    <thead class="table-dark">
                        <tr>
                            <th>Plate</th>
                            <th>Entry Time</th>
                            <th>Exit Time</th>
                            <th class="d-none d-md-table-cell">In (kg)</th>
                            <th class="d-none d-md-table-cell">Out (kg)</th>
                            <th class="d-none d-md-table-cell">Total (kg)</th>
                            <th>Large Image In</th>
                            <th class="d-none d-lg-table-cell">License Image In</th>
                            <th class="d-none d-lg-table-cell">Weight Image In</th>
                            <th class="d-none d-lg-table-cell">Large Image Out</th>
                            <th class="d-none d-lg-table-cell">License Image Out</th>
                            <th class="d-none d-lg-table-cell">Weight Image Out</th>
                            <!-- Removed Extra Column -->
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (!empty($pendingRows) && is_array($pendingRows)): ?>
                            <?php foreach ($pendingRows as $row): ?>
                                <?php
                                    $weight_diff = $row['in_weight'] - $row['out_weight'];
                                    $isUnknown = (strtolower($row['license_plate']) === 'unknown');
                                    $image1 = $row['image1'] ?? null;
                                    $image2 = $row['image2'] ?? null;
                                    $image3 = $row['image3'] ?? null;
                                    $out_image = $row['out_image'] ?? null;
                                    $out_image2 = $row['out_image2'] ?? null;
                                    $out_image3 = $row['out_image3'] ?? null;
                                ?>
                                <tr class="<?php echo $isUnknown ? 'unknown-row' : ''; ?>">
                                    <td>
                                        <?php if($isUnknown): ?>
                                            <input type="text" name="manual_plate_<?php echo htmlspecialchars($row['in_id']); ?>" 
                                                   id="manual_plate_<?php echo htmlspecialchars($row['in_id']); ?>" 
                                                   placeholder="Enter Plate" required 
                                                   class="form-control form-control-sm mb-2">
                                            <button type="button" onclick="updatePlate('<?php echo htmlspecialchars($row['in_id']); ?>','<?php echo htmlspecialchars($row['out_id']); ?>')" 
                                                    class="btn btn-success btn-sm mb-2">Save Changes</button>
                                            <div id="update_msg_<?php echo htmlspecialchars($row['in_id']); ?>" class="text-danger small"></div>
                                        <?php else: ?>
                                            <?php echo htmlspecialchars($row['license_plate']); ?>
                                        <?php endif; ?>
                                    </td>
                                    <td><?php echo htmlspecialchars($row['in_time']); ?></td>
                                    <td><?php echo htmlspecialchars($row['out_time']); ?></td>
                                    <td class="d-none d-md-table-cell"><?php echo htmlspecialchars($row['in_weight']); ?></td>
                                    <td class="d-none d-md-table-cell"><?php echo htmlspecialchars($row['out_weight']); ?></td>
                                    <td class="d-none d-md-table-cell"><?php echo htmlspecialchars($weight_diff); ?></td>
                                    <td>
                                        <?php if ($image1): ?>
                                            <img src="<?php echo htmlspecialchars($image1); ?>" alt="Large Image In" class="image-thumbnail img-fluid" onclick="openModal('<?php echo htmlspecialchars($image1); ?>')">
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </td>
                                    <td class="d-none d-lg-table-cell">
                                        <?php if ($image2): ?>
                                            <img src="<?php echo htmlspecialchars($image2); ?>" alt="License Image In" class="image-thumbnail img-fluid" onclick="openModal('<?php echo htmlspecialchars($image2); ?>')">
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </td>
                                    <td class="d-none d-lg-table-cell">
                                        <?php if ($image3): ?>
                                            <img src="<?php echo htmlspecialchars($image3); ?>" alt="Weight Image In" class="image-thumbnail img-fluid" onclick="openModal('<?php echo htmlspecialchars($image3); ?>')">
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </td>
                                    <td class="d-none d-lg-table-cell">
                                        <?php if ($out_image): ?>
                                            <img src="<?php echo htmlspecialchars($out_image); ?>" alt="Large Image Out" class="image-thumbnail img-fluid" onclick="openModal('<?php echo htmlspecialchars($out_image); ?>')">
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </td>
                                    <td class="d-none d-lg-table-cell">
                                        <?php if ($out_image2): ?>
                                            <img src="<?php echo htmlspecialchars($out_image2); ?>" alt="License Image Out" class="image-thumbnail img-fluid" onclick="openModal('<?php echo htmlspecialchars($out_image2); ?>')">
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </td>
                                    <td class="d-none d-lg-table-cell">
                                        <?php if ($out_image3): ?>
                                            <img src="<?php echo htmlspecialchars($out_image3); ?>" alt="Weight Image Out" class="image-thumbnail img-fluid" onclick="openModal('<?php echo htmlspecialchars($out_image3); ?>')">
                                        <?php else: ?>
                                            N/A
                                        <?php endif; ?>
                                    </td>
                                    <!-- Removed 'Rubbish Type & Amount' Column -->
                                    <td>
                                        <form method="post" action="pay.php" onsubmit="return validateForm('<?php echo htmlspecialchars($row['in_id']); ?>');">
                                            <select class="form-select form-select-sm mb-2" name="rubbish_type_id" required onchange="updateTotalAmount(this, <?php echo (float)$weight_diff; ?>, '<?php echo htmlspecialchars($row['in_id']); ?>')"
                                                <?php echo $isUnknown ? 'disabled' : ''; ?>>
                                                <option value="" disabled selected>Select...</option>
                                                <?php foreach ($types as $t): ?>
                                                    <option value="<?php echo htmlspecialchars($t['id']); ?>"><?php echo htmlspecialchars($t['name']); ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                            <div class="small" id="total-amount-<?php echo $row['in_id']; ?>"></div>
                                            <input type="hidden" name="in_id" value="<?php echo htmlspecialchars($row['in_id']); ?>">
                                            <input type="hidden" name="out_id" value="<?php echo htmlspecialchars($row['out_id']); ?>">
                                            <input type="hidden" name="weight_diff" value="<?php echo htmlspecialchars($weight_diff); ?>">
                                            <?php if($isUnknown): ?>
                                                <input type="hidden" name="manual_plate" id="hidden_manual_plate_<?php echo htmlspecialchars($row['in_id']); ?>" value="">
                                                <button type="submit" class="btn btn-primary btn-sm" disabled>Pay</button>
                                            <?php else: ?>
                                                <button type="submit" class="btn btn-primary btn-sm">Pay</button>
                                            <?php endif; ?>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="13" class="text-center">No pending payments found.</td> <!-- Adjusted colspan -->
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            <?php if($pending_total_pages > 1): ?>
            <nav aria-label="Pending pagination">
                <ul class="pagination justify-content-center">
                    <?php if($pending_page > 1): ?>
                        <li class="page-item">
                            <a class="page-link" href="?search_pending=<?php echo urlencode($search_pending); ?>&pending_page=<?php echo $pending_page-1; ?>#pending" aria-label="Previous">
                                <span aria-hidden="true">&laquo;</span>
                            </a>
                        </li>
                    <?php else: ?>
                        <li class="page-item disabled"><span class="page-link">&laquo;</span></li>
                    <?php endif; ?>
                    <?php
                    $max_links = 5;
                    $start = max($pending_page - floor($max_links / 2), 1);
                    $end = min($start + $max_links - 1, $pending_total_pages);
                    $start = max($end - $max_links + 1, 1);
                    for($i = $start; $i <= $end; $i++): ?>
                        <li class="page-item <?php if($i == $pending_page) echo 'active'; ?>">
                            <a class="page-link" href="?search_pending=<?php echo urlencode($search_pending); ?>&pending_page=<?php echo $i; ?>#pending"><?php echo $i; ?></a>
                        </li>
                    <?php endfor; ?>
                    <?php if($pending_page < $pending_total_pages): ?>
                        <li class="page-item">
                            <a class="page-link" href="?search_pending=<?php echo urlencode($search_pending); ?>&pending_page=<?php echo $pending_page+1; ?>#pending" aria-label="Next">
                                <span aria-hidden="true">&raquo;</span>
                            </a>
                        </li>
                    <?php else: ?>
                        <li class="page-item disabled"><span class="page-link">&raquo;</span></li>
                    <?php endif; ?>
                </ul>
            </nav>
            <?php endif; ?>
        </div>

        <!-- Transactions Tab -->
        <div class="tab-pane fade" id="transactions" role="tabpanel" aria-labelledby="transactions-tab">
            <div class="mt-3 mb-3 d-flex justify-content-between align-items-center flex-wrap">
                <form method="get" action="interface.php" class="d-flex flex-wrap">
                    <input type="text" name="search" class="form-control me-2 mb-2" placeholder="Search by License Plate" value="<?php echo htmlspecialchars($search); ?>">
                    <select name="rubbish_type_id" class="form-select me-2 mb-2" style="width: 200px;">
                        <option value="">All Rubbish Types</option>
                        <?php foreach ($types as $t): ?>
                            <option value="<?php echo htmlspecialchars($t['id']); ?>" <?php if($rubbish_type_id == $t['id']) echo 'selected'; ?>>
                                <?php echo htmlspecialchars($t['name']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <input type="date" name="start_date" class="form-control me-2 mb-2" placeholder="Start Date" value="<?php echo htmlspecialchars($start_date); ?>">
                    <input type="date" name="end_date" class="form-control me-2 mb-2" placeholder="End Date" value="<?php echo htmlspecialchars($end_date); ?>">
                    <button type="submit" class="btn btn-outline-success me-2 mb-2">Search</button>
                    <a href="interface.php" class="btn btn-secondary mb-2">Reset</a>
                </form>
                <form method="get" action="export_transactions.php">
                    <input type="hidden" name="search" value="<?php echo htmlspecialchars($search); ?>">
                    <input type="hidden" name="rubbish_type_id" value="<?php echo htmlspecialchars($rubbish_type_id); ?>">
                    <input type="hidden" name="start_date" value="<?php echo htmlspecialchars($start_date); ?>">
                    <input type="hidden" name="end_date" value="<?php echo htmlspecialchars($end_date); ?>">
                    <button type="submit" class="btn btn-outline-primary mb-2">Export to CSV</button>
                </form>
            </div>
            <div class="table-responsive">
                <table class="table table-bordered table-hover align-middle">
                    <thead class="table-dark">
                        <tr>
                            <th>Payment ID</th>
                            <th>License Plate</th>
                            <th>Rubbish Type</th>
                            <th>Weight Difference (kg)</th>
                            <th>Rate per kg</th>
                            <th>Total Amount (kr)</th>
                            <th>Payment Time</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if(empty($transactionRows)): ?>
                            <tr>
                                <td colspan="7" class="text-center">No transactions found.</td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($transactionRows as $trx): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($trx['payment_id']); ?></td>
                                <td><?php echo htmlspecialchars($trx['license_plate']); ?></td>
                                <td><?php echo htmlspecialchars($trx['rubbish_type']); ?></td>
                                <td><?php echo htmlspecialchars($trx['weight_diff']); ?></td>
                                <td><?php echo htmlspecialchars(number_format($trx['rate_per_kg'], 2)); ?></td>
                                <td><?php echo htmlspecialchars(number_format($trx['total_amount'], 2)); ?></td>
                                <td><?php echo htmlspecialchars($trx['payment_time']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
            <!-- Pagination -->
            <?php if($totalPages > 1): ?>
            <nav aria-label="Transaction pagination">
                <ul class="pagination justify-content-center">
                    <?php if($page > 1): ?>
                        <li class="page-item">
                            <a class="page-link" href="?search=<?php echo urlencode($search); ?>&rubbish_type_id=<?php echo urlencode($rubbish_type_id); ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&page=<?php echo $page-1; ?>" aria-label="Previous">
                                <span aria-hidden="true">&laquo;</span>
                            </a>
                        </li>
                    <?php else: ?>
                        <li class="page-item disabled">
                            <span class="page-link">&laquo;</span>
                        </li>
                    <?php endif; ?>

                    <?php
                    // Display limited page links
                    $max_links = 5;
                    $start = max($page - floor($max_links / 2), 1);
                    $end = min($start + $max_links - 1, $totalPages);
                    $start = max($end - $max_links + 1, 1);
                    for($i = $start; $i <= $end; $i++): ?>
                        <li class="page-item <?php if($i == $page) echo 'active'; ?>">
                            <a class="page-link" href="?search=<?php echo urlencode($search); ?>&rubbish_type_id=<?php echo urlencode($rubbish_type_id); ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&page=<?php echo $i; ?>">
                                <?php echo $i; ?>
                            </a>
                        </li>
                    <?php endfor; ?>

                    <?php if($page < $totalPages): ?>
                        <li class="page-item">
                            <a class="page-link" href="?search=<?php echo urlencode($search); ?>&rubbish_type_id=<?php echo urlencode($rubbish_type_id); ?>&start_date=<?php echo urlencode($start_date); ?>&end_date=<?php echo urlencode($end_date); ?>&page=<?php echo $page+1; ?>" aria-label="Next">
                                <span aria-hidden="true">&raquo;</span>
                            </a>
                        </li>
                    <?php else: ?>
                        <li class="page-item disabled">
                            <span class="page-link">&raquo;</span>
                        </li>
                    <?php endif; ?>
                </ul>
            </nav>
            <?php endif; ?>
        </div>
        <!-- POS Transactions Tab -->
        <div class="tab-pane fade" id="pos-trans" role="tabpanel" aria-labelledby="pos-trans-tab">
            <div class="table-responsive mt-3">
                <table class="table table-bordered table-hover align-middle">
                    <thead class="table-dark">
                        <tr>
                            <th>ID</th>
                            <th>Payment ID</th>
                            <th>License Plate</th>
                            <th>HTTP Status</th>
                            <th>Created At</th>
                            <th>Request</th>
                            <th>Response</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if(empty($posRows)): ?>
                            <tr><td colspan="7" class="text-center">No POS transactions found.</td></tr>
                        <?php else: foreach($posRows as $row): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($row['id']); ?></td>
                                <td><?php echo htmlspecialchars($row['payment_id']); ?></td>
                                <td><?php echo htmlspecialchars($row['license_plate'] ?? ''); ?></td>
                                <td><?php echo htmlspecialchars($row['http_status']); ?></td>
                                <td><?php echo htmlspecialchars($row['created_at']); ?></td>
                                <td class="json-cell"><?php echo htmlspecialchars($row['request_json']); ?></td>
                                <td class="json-cell"><?php echo htmlspecialchars($row['response_json']); ?></td>
                            </tr>
                        <?php endforeach; endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
        <!-- Statistics Tab -->
        <div class="tab-pane fade" id="stats" role="tabpanel" aria-labelledby="stats-tab">
            <div class="mt-3 mb-3">
                <form method="get" action="interface.php" class="row g-2 flex-wrap">
                    <div class="col-md-auto">
                        <input type="text" name="stats_plate" class="form-control" placeholder="License Plate" value="<?php echo htmlspecialchars($stats_plate); ?>">
                    </div>
                    <div class="col-md-auto">
                        <input type="date" name="stats_start_date" class="form-control" value="<?php echo htmlspecialchars($stats_start_date); ?>">
                    </div>
                    <div class="col-md-auto">
                        <input type="date" name="stats_end_date" class="form-control" value="<?php echo htmlspecialchars($stats_end_date); ?>">
                    </div>
                    <div class="col-md-auto">
                        <input type="time" name="stats_start_time" class="form-control" value="<?php echo htmlspecialchars($stats_start_time); ?>">
                    </div>
                    <div class="col-md-auto">
                        <input type="time" name="stats_end_time" class="form-control" value="<?php echo htmlspecialchars($stats_end_time); ?>">
                    </div>
                    <div class="col-md-auto">
                        <button type="submit" class="btn btn-outline-success">Search</button>
                        <a href="interface.php" class="btn btn-secondary">Reset</a>
                    </div>
                </form>
            </div>
            <div class="table-responsive">
                <table class="table table-bordered table-hover align-middle">
                    <thead class="table-dark">
                        <tr>
                            <th>Date</th>
                            <th>Total Cars</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if(empty($statsRows)): ?>
                            <tr><td colspan="2" class="text-center">No records found.</td></tr>
                        <?php else: foreach($statsRows as $row): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($row['day']); ?></td>
                                <td><?php echo htmlspecialchars($row['count']); ?></td>
                            </tr>
                        <?php endforeach; endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Modal Structure -->
    <div class="modal fade" id="imageModal" tabindex="-1" aria-labelledby="imageModalLabel" aria-hidden="true">
      <div class="modal-dialog modal-dialog-centered modal-lg">
        <div class="modal-content bg-dark">
          <div class="modal-header border-0">
            <h5 class="modal-title text-white" id="imageModalLabel">Image Preview</h5>
            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Close"></button>
          </div>
          <div class="modal-body text-center">
            <img src="" alt="Image" class="modal-img img-fluid">
          </div>
        </div>
      </div>
    </div>

    <!-- Bootstrap JS Bundle (includes Popper) -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Function to open image in modal
        function openModal(src) {
            const modal = new bootstrap.Modal(document.getElementById('imageModal'));
            const modalImg = document.querySelector('.modal-img');
            modalImg.src = src;
            modal.show();
        }

        // Function to update total amount based on rubbish type selection
        function updateTotalAmount(selectEl, weight_diff, in_id) {
            const selectedType = selectEl.value;
            const rubbishRates = <?php echo json_encode($typeRates); ?>;
            if (selectedType && rubbishRates[selectedType]) {
                const rate = rubbishRates[selectedType];
                const total = (weight_diff * rate).toFixed(2);
                document.getElementById('total-amount-' + in_id).textContent = "Total: kr." + total;
            } else {
                document.getElementById('total-amount-' + in_id).textContent = "";
            }
        }

        // Function to validate form before submission
        function validateForm(in_id) {
            const plateInput = document.getElementById('manual_plate_' + in_id);
            if (plateInput) {
                const manualPlate = plateInput.value.trim();
                if (!manualPlate) {
                    alert("Please enter a license plate.");
                    return false;
                }
                document.getElementById('hidden_manual_plate_' + in_id).value = manualPlate;
            }
            return true;
        }

        // Function to handle plate updates via AJAX
        function updatePlate(in_id, out_id) {
            const plateInput = document.getElementById('manual_plate_' + in_id);
            const newPlate = plateInput.value.trim();
            const msgDiv = document.getElementById('update_msg_' + in_id);
            msgDiv.textContent = '';

            if (!newPlate) {
                msgDiv.textContent = 'Plate cannot be empty.';
                return;
            }

            const formData = new FormData();
            formData.append('in_id', in_id);
            formData.append('out_id', out_id);
            formData.append('new_plate', newPlate);

            fetch('update_plate.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.status === 'success') {
                    // Optionally, display a success message without reloading
                    // For now, reload the page to reflect changes
                    window.location.href = 'interface.php?msg=Plate updated successfully.';
                } else {
                    msgDiv.textContent = data.message;
                }
            })
            .catch(err => {
                console.error(err);
                msgDiv.textContent = 'Error updating plate.';
            });
        }

        // **Real-Time Updates via AJAX Polling**
        const pendingPage = <?php echo (int)$pending_page; ?>;
        setInterval(fetchPendingPayments, 5000); // Fetch every 5 seconds

        function fetchPendingPayments() {
            const params = new URLSearchParams();
            params.append('fetch_pending', '1');
            params.append('pending_page', pendingPage.toString());
            <?php if (!empty($search_pending)): ?>
                params.append('search_pending', '<?php echo htmlspecialchars($search_pending); ?>');
            <?php endif; ?>
            fetch('fetch_pending.php?' + params.toString())
                .then(response => response.json())
                .then(data => {
                    if (data.status === 'success') {
                        const tbody = document.querySelector('#pendingPaymentsTable tbody');
                        tbody.innerHTML = ''; // Clear existing rows
                        data.data.forEach(row => {
                            const tr = document.createElement('tr');
                            if (row.license_plate.toLowerCase() === 'unknown') {
                                tr.classList.add('unknown-row');
                            }

                            // Calculate weight difference
                            const weight_diff = row.in_weight - row.out_weight;

                            // Create table cells
                            tr.innerHTML = `
                                <td>
                                    ${row.license_plate.toLowerCase() === 'unknown' ? `
                                        <input type="text" name="manual_plate_${row.in_id}" id="manual_plate_${row.in_id}" placeholder="Enter Plate" required class="form-control form-control-sm mb-2">
                                        <button type="button" onclick="updatePlate('${row.in_id}','${row.out_id}')" class="btn btn-success btn-sm mb-2">Save Changes</button>
                                        <div id="update_msg_${row.in_id}" class="text-danger small"></div>
                                    ` : `
                                        ${row.license_plate}
                                    `}
                                </td>
                                <td>${row.in_time}</td>
                                <td>${row.out_time}</td>
                                <td class="d-none d-md-table-cell">${row.in_weight}</td>
                                <td class="d-none d-md-table-cell">${row.out_weight}</td>
                                <td class="d-none d-md-table-cell">${weight_diff}</td>
                                <td>
                                    ${row.image1 ? `<img src="${row.image1}" alt="Large Image In" class="image-thumbnail img-fluid" onclick="openModal('${row.image1}')">` : 'N/A'}
                                </td>
                                <td class="d-none d-lg-table-cell">
                                    ${row.image2 ? `<img src="${row.image2}" alt="License Image In" class="image-thumbnail img-fluid" onclick="openModal('${row.image2}')">` : 'N/A'}
                                </td>
                                <td class="d-none d-lg-table-cell">
                                    ${row.image3 ? `<img src="${row.image3}" alt="Weight Image In" class="image-thumbnail img-fluid" onclick="openModal('${row.image3}')">` : 'N/A'}
                                </td>
                                <td class="d-none d-lg-table-cell">
                                    ${row.out_image ? `<img src="${row.out_image}" alt="Large Image Out" class="image-thumbnail img-fluid" onclick="openModal('${row.out_image}')">` : 'N/A'}
                                </td>
                                <td class="d-none d-lg-table-cell">
                                    ${row.out_image2 ? `<img src="${row.out_image2}" alt="License Image Out" class="image-thumbnail img-fluid" onclick="openModal('${row.out_image2}')">` : 'N/A'}
                                </td>
                                <td class="d-none d-lg-table-cell">
                                    ${row.out_image3 ? `<img src="${row.out_image3}" alt="Weight Image Out" class="image-thumbnail img-fluid" onclick="openModal('${row.out_image3}')">` : 'N/A'}
                                </td>
                                <td>
                                    <form method="post" action="pay.php" onsubmit="return validateForm('${row.in_id}');">
                                        <select class="form-select form-select-sm mb-2" name="rubbish_type_id" required onchange="updateTotalAmount(this, ${weight_diff}, '${row.in_id}')"
                                            ${row.license_plate.toLowerCase() === 'unknown' ? 'disabled' : ''}>
                                            <option value="" disabled selected>Select...</option>
                                            <?php foreach ($types as $t): ?>
                                                <option value="<?php echo htmlspecialchars($t['id']); ?>"><?php echo htmlspecialchars($t['name']); ?></option>
                                            <?php endforeach; ?>
                                        </select>
                                        <div class="small" id="total-amount-${row.in_id}"></div>
                                        <input type="hidden" name="in_id" value="${row.in_id}">
                                        <input type="hidden" name="out_id" value="${row.out_id}">
                                        <input type="hidden" name="weight_diff" value="${weight_diff}">
                                        ${row.license_plate.toLowerCase() === 'unknown' ? `
                                            <input type="hidden" name="manual_plate" id="hidden_manual_plate_${row.in_id}" value="">
                                            <button type="submit" class="btn btn-primary btn-sm" disabled>Pay</button>
                                        ` : `
                                            <button type="submit" class="btn btn-primary btn-sm">Pay</button>
                                        `}
                                    </form>
                                </td>
                            `;
                            tbody.appendChild(tr);
                        });
                    }
                })
                .catch(err => {
                    console.error('Error fetching pending payments:', err);
                });
        }
    </script>
</body>
</html>
